// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * A type of dialogue used as for choosing modules in a course.
 *
 * @module     core_course/activitychooser
 * @copyright  2020 Mathew May <mathew.solutions>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

import * as ChooserDialogue from 'core_course/local/activitychooser/dialogue';
import * as Repository from 'core_course/local/activitychooser/repository';
import selectors from 'core_course/local/activitychooser/selectors';
import CustomEvents from 'core/custom_interaction_events';
import * as Templates from 'core/templates';
import * as ModalFactory from 'core/modal_factory';
import {get_string as getString} from 'core/str';
import Pending from 'core/pending';
import Ajax from 'core/ajax';
import {getLabel as getStringBooste} from './helper';
import {thirdFormLoader} from './helper';
import Notification from "core/notification";
// Set up some JS module wide constants that can be added to in the future.

// Tab config options.
const ALLACTIVITIESRESOURCES = 0;
const ONLYALL = 1;
const ACTIVITIESRESOURCES = 2;

// Module types.
//const ACTIVITY = 0;
const RESOURCE = 1;

/**
 * Set up the activity chooser.
 *
 * @method init
 * @param {Number} courseId Course ID to use later on in fetchModules()
 * @param {Number} boosteId The ID of the scenario created
 * @param {Object} chooserConfig Any PHP config settings that we may need to reference
 */
export const init = (courseId, boosteId, chooserConfig) => {
    const pendingPromise = new Pending();
    registerListenerEvents(courseId, boosteId, chooserConfig);

    pendingPromise.resolve();
    thirdFormLoader();
};


/**
 * Set up the suggested activities in third form.
 *
 * @method recommendations
 * @param {Number} courseId Course ID to use later on in fetchModules()
 * @param {Number} boosteId The ID of the scenario created
 * @param {Number} taxonomy Name of the taxonomy to parameter the recommendations
 * @param {Number} points number of points of the scenario to compare with the recommended values of activities
 */
export async function recommendations(courseId, boosteId, taxonomy = "all", points) {
    let bodyPromiseResolver;

    var moretext = await getStringBooste("more");
    var lesstext = await getStringBooste("less");

    const fetchModuleData = (() => {
        let innerPromise = null;
        return () => {
            if (!innerPromise) {
                innerPromise = new Promise((resolve) => {
                    resolve(Repository.activityModules(courseId));
                });
            }
            return innerPromise;
        };
    })();

    const fetchExtActivitiesData = (() => {
        let innerPromise = null;
        return () => {
            if (!innerPromise) {
                innerPromise = new Promise((resolve) => {
                    resolve(getExtActivities(courseId));
                });
            }
            return innerPromise;
        };
    })();

    const data = await fetchModuleData().catch(async(e) => {
        const errorTemplateData = {
            'errormessage': e.message
        };
        bodyPromiseResolver(await Templates.render('core_course/local/activitychooser/error', errorTemplateData));
    });

    data.content_items.map((d) => { d.activity_type = "int";});

    const eadata = await fetchExtActivitiesData().catch(async(e) => {
        const errorTemplateData = {
            'errormessage': e.message
        };
        bodyPromiseResolver(await Templates.render('core_course/local/activitychooser/error', errorTemplateData));
    });

    const fetchRecommendedData = (() => {
        let innerPromise = null;
        return () => {
            if (!innerPromise) {
                innerPromise = new Promise((resolve) => {
                    resolve(getRecommendedActivities(boosteId, taxonomy));
                });
            }
            return innerPromise;
        };
    })();

    data.content_items.push(...eadata);
    data.content_items.splice(Object.keys(data.content_items).find(key => data.content_items[key].name === "lti"), 1);

    const recommended = await fetchRecommendedData().catch(async(e) => {
        const errorTemplateData = {
            'errormessage': e.message
        };
        bodyPromiseResolver(await Templates.render('core_course/local/activitychooser/error', errorTemplateData));
    });

    const recommendedForTemplate = setRecommendedActivities(data.content_items, recommended, points);
    const params = {
        recommended : recommendedForTemplate,
        taxonomy : taxonomy,
        taxonomy_i18n : M.util.get_string(taxonomy, 'local_booste')
    };
    console.log("i18n", taxonomy, M.util.get_string(taxonomy, 'local_booste'));

    var idContainer = document.getElementById(taxonomy+"_suggested_activities");
    console.log("parameters", params);
    Templates.render('local_booste/activitysuggestedlist', params).then(function(html, js) {

        Templates.replaceNodeContents(idContainer, html, js);

        console.log("recommended.length" , recommended.length);
        if(recommended.length > 10) {

            $("#local-booste-recommended-"+taxonomy).after('<a href="#" id="display_recs_'+taxonomy+'">'+moretext+'</a>');
            var shownDefault = 10;
            var numShown = shownDefault; // Initial rows shown & index
            var $table = $('.tbl').find('tbody');  // tbody containing all the rows
            var numRows = $table.find('tr').length; // Total # rows

            $table.find('tr:gt(' + (numShown - 1) + ')').hide();
            $('#display_recs_'+taxonomy+'').on('click', function() {
                if (numShown === numRows) {
                // since we show all rows node, hiding some
                    numShown = shownDefault;
                    $table.find('tr:gt(' + (numShown - 1) + ')').hide();
                    $('#display_recs_'+taxonomy+'').text(moretext);
                } else {
                    numShown = numRows;
                    $('#display_recs_'+taxonomy+'').text(lesstext);
                }
                $table.find('tr:lt(' + numShown + ')').show();
            });
        }
    });
}

/**
 * Get ext activites by the tags of a scenario
 *
 * @method getExtActivities
 * @param {Number} courseId Course ID to use later on in fetchModules()
 * @return {Array}
 */
 const getExtActivities = (courseId) => {
    const request = {
        methodname: 'local_booste_get_ext_activities',
        args: {
            id: courseId
        }
    };
    return Ajax.call([request])[0];
};

/**
 * Get recommended activites by the tags of a scenario
 *
 * @method getRecommendedActivities
 * @param {int} boosteId booste id
 * @param {string} taxonomy taxonomy name
 * @return {Array}
 */
 const getRecommendedActivities = (boosteId, taxonomy) => {
    const request = {
        methodname: 'local_booste_get_recommendations',
        args: {
            id: boosteId,
            taxonomy: taxonomy
        }
    };
    return Ajax.call([request])[0];
};

/**
 * Set recommended activites to true in the list of activities
 *
 * @method setRecommendedActivities
 * @param {Array} data is all activities
 * @param {Array} context is the recommended activities
 * @param {Array} points is the points of the scenario, to compare with activities points
 * @return {Array}
 */
const setRecommendedActivities = (data, context, points) => {

    var recActivities = [];
    var extRecActivities = [];

    var index = 1;

    context.forEach(rec => {
        if(rec.ext_activity) {
            extRecActivities[rec.ext_activity] = rec.points;
        }
        else if(rec.activity) {
            recActivities[rec.activity] = rec.points;
        }
    });

    const extRecActivitiesKeys = Object.keys(extRecActivities);
    const recActivitiesKeys = Object.keys(recActivities);


    data.forEach(row => {
        row.recommended = false;
        row.recommended_value = 0;
        row.match = 0;
        //points = Math.ceil(points + 40);


        if(row.activity_type == "ext" && extRecActivitiesKeys.includes(row.id.toString())) {
            row.recommended = true;
            //row.recommended_value =  Math.ceil(row.recommended_value + parseInt(extRecActivities[row.id]));
            row.recommended_value = parseInt(extRecActivities[row.id] + 10);
        }
        else if(row.activity_type == "int" && recActivitiesKeys.includes(row.id.toString())) {
            row.recommended = true;
            //row.recommended_value =  Math.ceil(row.recommended_value + parseInt(recActivities[row.id]));
            row.recommended_value = parseInt(recActivities[row.id] + 10);
        }

        row.match = Math.ceil(row.recommended_value / points * 100);
    });

    var recommended = data.filter(mod => mod.recommended === true);
    recommended = recommended.sort((a,b) => b.recommended_value - a.recommended_value);
    recommended.forEach(row => {
        row.index = index++;
    });
    // if(recommended.length > 30) {
    //     recommended.length = 30;
    // }
    return recommended;
};



/**
 * Set link to label from the button create a guideline.
 *
 * @method createGuideline
 */
export const createGuideline = () => {
    let selectors = document.querySelectorAll('[data-action=add_guideline]');
    selectors.forEach(selectors => {
        selectors.addEventListener('click', e => {
            var data = e.target.dataset;
            var url = window.location.href;
            var urlParams = new URLSearchParams(window.location.search);
            var boosteid = urlParams.get('booste');
            var courseid = urlParams.get('course');

            url = url.split("booste_third_form")[0];
            url += 'modedit.php?&course='+courseid+'&add=label&booste='
                    + boosteid+'&booste_step='+data.id+'&section='+data.sectionid+'&sr=&archetype=1';
            window.location.href = url;
        });
    });
};

/**
 * Init chart of activities by taxonomy.
 *
 * @param {Number} boosteId The ID of the scenario created
 * @method createGuideline
 */
export async function initChart(boosteId) {

    const synthesis = await getStringBooste('synthesis');
    const evaluation = await getStringBooste('evaluation');
    const analysis = await getStringBooste('analysis');
    const application = await getStringBooste('application');
    const comprehension = await getStringBooste('comprehension');
    const knowledge = await getStringBooste('knowledge');

    const detailSynthesis = await getStringBooste('detail_synthesis');
    const detailEvaluation = await getStringBooste('detail_evaluation');
    const detailAnalysis = await getStringBooste('detail_analysis');
    const detailApplication = await getStringBooste('detail_application');
    const detailComprehension = await getStringBooste('detail_comprehension');
    const detailKnowledge = await getStringBooste('detail_knowledge');

    const title = await getStringBooste('taxonomy_chart');
    const objectivesNumber = await getStringBooste('objectives_number');

    document.getElementById("id_all").style.backgroundColor = "#e3e3e3";
    document.getElementById("id_synthesis").style.backgroundColor = "#92d050";
    document.getElementById("id_evaluation").style.backgroundColor = "#ff0066";
    document.getElementById("id_analysis").style.backgroundColor = "#00b0f0";
    document.getElementById("id_application").style.backgroundColor = "#ea6b14";
    document.getElementById("id_comprehension").style.backgroundColor = "#a162d0";
    document.getElementById("id_knowledge").style.backgroundColor = "#fcd304";

    var synthesisVal = 0;
    var evaluationVal = 0;
    var analysisVal = 0;
    var applicationVal = 0;
    var comprehensionVal = 0;
    var knowledgeVal = 0;

    const request = {
        methodname: 'local_booste_get_objectives',
        args: {
            id: boosteId
        }
    };

    Ajax.call([request])[0].done((data) => {

        document.getElementById("id_all").removeAttribute("hidden");
        data.forEach(objective => {
            switch(objective.taxonomy_name) {
                case 'synthesis':
                    synthesisVal++;
                    document.getElementById("id_synthesis").removeAttribute("hidden");
                    break;
                case 'evaluation':
                    evaluationVal++;
                    document.getElementById("id_evaluation").removeAttribute("hidden");
                    break;
                case 'analysis':
                    analysisVal++;
                    document.getElementById("id_analysis").removeAttribute("hidden");
                    break;
                case 'application':
                    applicationVal++;
                    document.getElementById("id_application").removeAttribute("hidden");
                    break;
                case 'comprehension':
                    comprehensionVal++;
                    document.getElementById("id_comprehension").removeAttribute("hidden");
                    break;
                case 'knowledge':
                    knowledgeVal++;
                    document.getElementById("id_knowledge").removeAttribute("hidden");
                    break;
            }
        });

        const chartdata = {
            "type": "pie",
            "series": [{
                "label": objectivesNumber,
                "labels":[detailKnowledge, detailComprehension, detailApplication, detailAnalysis,
                          detailEvaluation, detailSynthesis],
                "type":null,
                "values":[knowledgeVal, comprehensionVal, applicationVal, analysisVal, evaluationVal, synthesisVal],
                "colors": [],
                "axes":{"x":null,"y":null},
                "smooth":null
            }],
            "labels": [knowledge, comprehension, application, analysis, evaluation, synthesis],
            "title": title,
            "axes": {"x":[],"y":[]},
            "config_colorset": null,
            "doughnut": false
        };

        const dataForChart = {
            "chartdata": JSON.stringify(chartdata),
            "withtable": true
        };

        Templates.render('local_booste/activitychart', dataForChart).then(function(html, js) {
            Templates.replaceNodeContents(document.getElementById('taxonomy_chart'), html, js);
        });
    }).catch(Notification.exception);
}


/**
 * Once a selection has been made make the modal & module information and pass it along
 *
 * @method registerListenerEvents
 * @param {Number} courseId
 * @param {Number} boosteId The ID of the scenario created
 * @param {Object} chooserConfig Any PHP config settings that we may need to reference
 */
 const registerListenerEvents = (courseId, boosteId, chooserConfig) => {
    const events = [
        'click',
        CustomEvents.events.activate,
        CustomEvents.events.keyboardActivate
    ];

    const fetchModuleData = (() => {
        let innerPromise = null;

        return () => {
            if (!innerPromise) {
                innerPromise = new Promise((resolve) => {
                    resolve(Repository.activityModules(courseId));
                });
            }

            return innerPromise;
        };
    })();

    const fetchFooterData = (() => {
        let footerInnerPromise = null;

        return (sectionId) => {
            if (!footerInnerPromise) {
                footerInnerPromise = new Promise((resolve) => {
                    resolve(Repository.fetchFooterData(courseId, sectionId));
                });
            }

            return footerInnerPromise;
        };
    })();

    const fetchExtActivitiesData = (() => {
        let innerPromise = null;
        return () => {
            if (!innerPromise) {
                innerPromise = new Promise((resolve) => {
                    resolve(getExtActivities(courseId));
                });
            }
            return innerPromise;
        };
    })();

    const fetchRecommendedData = (() => {
        let innerPromise = null;
        return () => {
            if (!innerPromise) {
                innerPromise = new Promise((resolve) => {
                    resolve(getRecommendedActivities(boosteId, "all"));
                });
            }
            return innerPromise;
        };
    })();

    CustomEvents.define(document, events);

    // Display module chooser event listeners.
    events.forEach((event) => {
        document.addEventListener(event, async(e) => {
            if(e.target.dataset.action == "add_activity" || e.target.dataset.action == "add_resource") {
                let caller;
                caller = e.target;

                // We want to show the modal instantly but loading whilst waiting for our data.
                let bodyPromiseResolver;
                const bodyPromise = new Promise(resolve => {
                    bodyPromiseResolver = resolve;
                });

                const footerData = await fetchFooterData(caller.dataset.sectionid);
                const sectionModal = buildModal(bodyPromise, footerData);

                // Now we have a modal we should start fetching data.
                // If an error occurs while fetching the data, display the error within the modal.
                const data = await fetchModuleData().catch(async(e) => {
                    const errorTemplateData = {
                        'errormessage': e.message
                    };
                    bodyPromiseResolver(await Templates.render('core_course/local/activitychooser/error', errorTemplateData));
                });

                const extActivities = await fetchExtActivitiesData().catch(async(e) => {
                    const errorTemplateData = {
                        'errormessage': e.message
                    };
                    bodyPromiseResolver(await Templates.render('core_course/local/activitychooser/error', errorTemplateData));
                });

                extActivities.map((e) => { e.link_helper = e.link; });
                data.content_items.map((d) => { d.activity_type = "int"; });
                // Early return if there is no module data.
                if (!data) {
                    return;
                }
                data.content_items.push(...extActivities);

                const recommended = await fetchRecommendedData().catch(async(e) => {
                    const errorTemplateData = {
                        'errormessage': e.message
                    };
                    bodyPromiseResolver(await Templates.render('core_course/local/activitychooser/error', errorTemplateData));
                });

                setRecommendedActivities(data.content_items, recommended);

                // Apply the section id to all the module instance links.
                const builtModuleData = sectionIdMapper(data, boosteId, caller.dataset.id, caller.dataset.sectionid,
                                                                caller.dataset.sectionreturnid, courseId);
                ChooserDialogue.displayChooser(
                    sectionModal,
                    builtModuleData,
                    partiallyAppliedFavouriteManager(data, caller.dataset.sectionid),
                    footerData,
                );

                if(e.target.dataset.action == "add_activity") {
                    bodyPromiseResolver(await Templates.render(
                        'local_booste/activitychooser',
                        templateDataBuilder(builtModuleData, chooserConfig)
                    ));
                }
                else if (e.target.dataset.action == "add_resource") {
                    bodyPromiseResolver(await Templates.render(
                        'local_booste/resourcechooser',
                        templateDataBuilder(builtModuleData, chooserConfig)
                    ));
                }
            }
        });
    });
};

/**
 * Given the web service data and an ID we want to make a deep copy
 * of the WS data then add on the section ID to the addoption URL
 *
 * @method sectionIdMapper
 * @param {Object} webServiceData Our original data from the Web service call
 * @param {Number} boosteId The ID of the scenario created
 * @param {Number} boosteStep The ID of a step of the scenario created
 * @param {Number} id The ID of the section we need to append to the links
 * @param {Number|null} sectionreturnid The ID of the section return we need to append to the links
 * @param {Number} courseId The ID of the actual course
 * @return {Array} [modules] with URL's built
 */
const sectionIdMapper = (webServiceData, boosteId, boosteStep, id, sectionreturnid, courseId) => {
    // We need to take a fresh deep copy of the original data as an object is a reference type.
    const newData = JSON.parse(JSON.stringify(webServiceData));
    newData.content_items.forEach((module) => {
        if(module.activity_type == "int") {
            module.link = module.link.replace("id=", "course=");
            if(module.componentname == "mod_game") {
                module.link = module.link.replace("course/mod", "local/booste/forms/mod");
            }
            else {
                module.link = module.link.replace("course/mod", "local/booste/forms/modedit");
            }
            module.link += '&booste=' + boosteId + '&booste_step=' + boosteStep +'&section=' + id
                                    + '&sr=' + (sectionreturnid ?? 0) + '&archetype=' + module.archetype ;
        }
        else if(module.activity_type == "ext")  {
            var link = window.location.href.split("forms/");
            module.link = link[0] + "forms/modedit.php";
            module.link += '?course=' + courseId + '&add=url&booste=' + boosteId + '&booste_step=' + boosteStep
                            + '&section=' + id + '&sr=' + (sectionreturnid ?? 0) + '&archetype=' + module.archetype
                            + '&ext_activity=' + module.id;
        }
    });
    return newData.content_items;
};

/**
 * Given an array of modules we want to figure out where & how to place them into our template object
 *
 * @method templateDataBuilder
 * @param {Array} data our modules to manipulate into a Templatable object
 * @param {Object} chooserConfig Any PHP config settings that we may need to reference
 * @return {Object} Our built object ready to render out
 */
const templateDataBuilder = (data, chooserConfig) => {
    // Setup of various bits and pieces we need to mutate before throwing it to the wolves.
    let activities = [];
    let ext_activities = [];
    let resources = [];
    let showAll = true;
    let showActivities = false;
    let showResources = false;

    // Tab mode can be the following [All, Resources & Activities, All & Activities & Resources].
    const tabMode = parseInt(chooserConfig.tabmode);

    // Filter the incoming data to find favourite & recommended modules.
    const favourites = data.filter(mod => mod.favourite === true);
    const recommended = data.filter(mod => mod.recommended === true);


    // Both of these modes need Activity & Resource tabs.
    if ((tabMode === ALLACTIVITIESRESOURCES || tabMode === ACTIVITIESRESOURCES) && tabMode !== ONLYALL) {
        // Filter the incoming data to find activities then resources.
        activities = data.filter(mod => mod.activity_type === "int");
        ext_activities = data.filter(mod => mod.activity_type === "ext");
        resources = data.filter(mod => mod.archetype === RESOURCE);
        showActivities = true;
        showResources = true;

        // We want all of the previous information but no 'All' tab.
        if (tabMode === ACTIVITIESRESOURCES) {
            showAll = false;
        }
    }
    // Given the results of the above filters lets figure out what tab to set active.
    // We have some favourites.
    const favouritesFirst = !!favourites.length;
    // We are in tabMode 2 without any favourites.
    const activitiesFirst = showAll === false && favouritesFirst === false;
    // We have nothing fallback to show all modules.
    const fallback = showAll === true && favouritesFirst === false;

    return {
        'default': data,
        showAll: showAll,
        activities: activities,
        ext_activities: ext_activities,
        showActivities: showActivities,
        activitiesFirst: activitiesFirst,
        resources: resources,
        showResources: showResources,
        favourites: favourites,
        recommended: recommended,
        favouritesFirst: favouritesFirst,
        fallback: fallback,
    };
};

/**
 * Given an object we want to build a modal ready to show
 *
 * @method buildModal
 * @param {Promise} bodyPromise
 * @param {String|Boolean} footer Either a footer to add or nothing
 * @return {Object} The modal ready to display immediately and render body in later.
 */
const buildModal = (bodyPromise, footer) => {
    return ModalFactory.create({
        type: ModalFactory.types.DEFAULT,
        title: getString('addresourceoractivity'),
        body: bodyPromise,
        footer: footer.customfootertemplate,
        large: true,
        scrollable: false,
        templateContext: {
            classes: 'modchooser'
        }
    })
    .then(modal => {
        modal.show();
        return modal;
    });
};

/**
 * A small helper function to handle the case where there are no more favourites
 * and we need to mess a bit with the available tabs in the chooser
 *
 * @method nullFavouriteDomManager
 * @param {HTMLElement} favouriteTabNav Dom node of the favourite tab nav
 * @param {HTMLElement} modalBody Our current modals' body
 */
const nullFavouriteDomManager = (favouriteTabNav, modalBody) => {
    favouriteTabNav.tabIndex = -1;
    favouriteTabNav.classList.add('d-none');
    // Need to set active to an available tab.
    if (favouriteTabNav.classList.contains('active')) {
        favouriteTabNav.classList.remove('active');
        favouriteTabNav.setAttribute('aria-selected', 'false');
        const favouriteTab = modalBody.querySelector(selectors.regions.favouriteTab);
        favouriteTab.classList.remove('active');
        const defaultTabNav = modalBody.querySelector(selectors.regions.defaultTabNav);
        const activitiesTabNav = modalBody.querySelector(selectors.regions.activityTabNav);
        if (defaultTabNav.classList.contains('d-none') === false) {
            defaultTabNav.classList.add('active');
            defaultTabNav.setAttribute('aria-selected', 'true');
            defaultTabNav.tabIndex = 0;
            defaultTabNav.focus();
            const defaultTab = modalBody.querySelector(selectors.regions.defaultTab);
            defaultTab.classList.add('active');
        } else {
            activitiesTabNav.classList.add('active');
            activitiesTabNav.setAttribute('aria-selected', 'true');
            activitiesTabNav.tabIndex = 0;
            activitiesTabNav.focus();
            const activitiesTab = modalBody.querySelector(selectors.regions.activityTab);
            activitiesTab.classList.add('active');
        }

    }
};

/**
 * Export a curried function where the builtModules has been applied.
 * We have our array of modules so we can rerender the favourites area and have all of the items sorted.
 *
 * @method partiallyAppliedFavouriteManager
 * @param {Array} moduleData This is our raw WS data that we need to manipulate
 * @param {Number} sectionId We need this to add the sectionID to the URL's in the faves area after rerender
 * @return {Function} partially applied function so we can manipulate DOM nodes easily & update our internal array
 */
const partiallyAppliedFavouriteManager = (moduleData, sectionId) => {
    /**
     * Curried function that is being returned.
     *
     * @param {String} internal Internal name of the module to manage
     * @param {Boolean} favourite Is the caller adding a favourite or removing one?
     * @param {HTMLElement} modalBody What we need to update whilst we are here
     */
    return async(internal, favourite, modalBody) => {
        const favouriteArea = modalBody.querySelector(selectors.render.favourites);

        // eslint-disable-next-line max-len
        const favouriteButtons = modalBody.querySelectorAll(`[data-internal="${internal}"] ${selectors.actions.optionActions.manageFavourite}`);
        const favouriteTabNav = modalBody.querySelector(selectors.regions.favouriteTabNav);
        const result = moduleData.content_items.find(({name}) => name === internal);
        const newFaves = {};
        if (result) {
            if (favourite) {
                result.favourite = true;

                // eslint-disable-next-line camelcase
                newFaves.content_items = moduleData.content_items.filter(mod => mod.favourite === true);

                const builtFaves = sectionIdMapper(newFaves, sectionId);

                const {html, js} = await Templates.renderForPromise('core_course/local/activitychooser/favourites',
                    {favourites: builtFaves});

                await Templates.replaceNodeContents(favouriteArea, html, js);

                Array.from(favouriteButtons).forEach((element) => {
                    element.classList.remove('text-muted');
                    element.classList.add('text-primary');
                    element.dataset.favourited = 'true';
                    element.setAttribute('aria-pressed', true);
                    element.firstElementChild.classList.remove('fa-star-o');
                    element.firstElementChild.classList.add('fa-star');
                });

                favouriteTabNav.classList.remove('d-none');
            } else {
                result.favourite = false;

                const nodeToRemove = favouriteArea.querySelector(`[data-internal="${internal}"]`);

                nodeToRemove.parentNode.removeChild(nodeToRemove);

                Array.from(favouriteButtons).forEach((element) => {
                    element.classList.add('text-muted');
                    element.classList.remove('text-primary');
                    element.dataset.favourited = 'false';
                    element.setAttribute('aria-pressed', false);
                    element.firstElementChild.classList.remove('fa-star');
                    element.firstElementChild.classList.add('fa-star-o');
                });
                const newFaves = moduleData.content_items.filter(mod => mod.favourite === true);

                if (newFaves.length === 0) {
                    nullFavouriteDomManager(favouriteTabNav, modalBody);
                }
            }
        }
    };
};

