// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Lib of js functions for booste
 *
 * @module     core_course/activitychooser
 * @copyright  2020 Mathew May <mathew.solutions>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

import {get_string as getString} from 'core/str';
import Ajax from "core/ajax";
import Notification from "core/notification";
import * as Templates from 'core/templates';

/**
 * Getting verbs for the taxonomy choosen
 *
 * @method getVerbs
 * @param {Number} taxonomyid
 */
export const getVerbs = (taxonomyid) => {
    fetchVerbs(taxonomyid);
    document.querySelector('[name=taxonomy]').addEventListener('change', (event) => {
        fetchVerbs(event.target.value);
    });
    var objphrase = document.getElementById("objective_example_1").innerHTML;
    objphrase = objphrase.replaceAll('[', '<span class="objective-verb-1">[');
    objphrase = objphrase.replaceAll(']', ']</span>');
    document.getElementById("objective_example_1").innerHTML = objphrase;

    var objphrase2 = document.getElementById("objective_example_2").innerHTML;
    objphrase2 = objphrase2.replaceAll('[', '<span class="objective-verb-2">[');
    objphrase2 = objphrase2.replaceAll(']', ']</span>');
    document.getElementById("objective_example_2").innerHTML = objphrase2;

};

/**
 * Fetching verbs for the taxonomy choosen
 *
 * @method fetchVerbs
 * @param {Number} taxonomyid
 */
const fetchVerbs = (taxonomyid) => {
    const request = {
        methodname: 'local_booste_get_verbs',
        args: {
            taxonomyid: taxonomyid
        }
    };
    Ajax.call([request])[0].done(function(element){
        document.getElementById("verbs_list").innerHTML = '';
        element.forEach(e => {
            document.getElementById("verbs_list")
                .innerHTML += '<span class="badge badge-info mb-3 mr-1" style="font-size: 100%">'+e.name+'</span>';
        });
    }).catch(Notification.exception);
};

/**
 * Get label of a string via get_string
 *
 * @method getLabel
 * @param {String} string
 */
export async function getLabel(string) {
    let bodyPromiseResolver;
    const fetchLabel = (() => {
        let innerPromise = null;
        return () => {
            if (!innerPromise) {
                innerPromise = new Promise((resolve) => {
                    resolve(getStringBooste(string));
                });
            }
            return innerPromise;
        };
    })();

    const label = await fetchLabel().catch(async(e) => {
        const errorTemplateData = {
            'errormessage': e.message
        };
        bodyPromiseResolver(await Templates.render('core_course/local/activitychooser/error', errorTemplateData));
    });
    return label;
}

/**
 * Get string via get_string
 *
 * @method getStringBooste
 * @param {String} name
 */
const getStringBooste = (name) => {
    return getString(name, 'local_booste').then(string => {
        return string;
    }).catch();
};

/**
 * Display helper of an activity
 *
 * @method getActivityHelp
 */
export async function getActivityHelp() {

    let helpers = document.querySelectorAll('[data-action=activity_helper]');
    helpers.forEach(helper => {
        helper.addEventListener('click', e => {
            e.preventDefault();
            const dataset = e.currentTarget.dataset;
            const id = dataset.id;
            const type = dataset.activitytype;
            const request = {
                methodname: 'local_booste_get_activity_helper',
                args: {
                    id: id,
                    type: type
                }
            };
            Ajax.call([request])[0].done(function(element){
                var popupid = document.getElementById('activity_helper');
                Templates.render('local_booste/activityhelper', element).then(function(html, js) {
                    Templates.replaceNodeContents(popupid, html, js);
                });
            }).catch(Notification.exception);
        });
    });
}

/**
 * Change submit buttons for activity setting in a scenario
 *
 * @method redoActivityButtons
 * @param {String} label
 */
export const redoActivityButtons = (label) => {
    document.getElementById('id_submitbutton2').value = label;
    document.getElementById('id_submitbutton').hidden = true;

    window.console.log($("[id^=single_button]").first());

};

/**
 * Save order of elements in lists and steps of a scenario
 *
 * @method saveOrder
 * @param {Object} e
 */
export const saveOrder = (e) => {
    var reorderlist = [];
    var target = e.target.dataset;

    document.getElementsByName("elements_list").forEach(function(e) {
        var elements = e.firstElementChild.children;
        var count = 1;
        var stepid = e.id.replace('step_', '');
        var stepid = stepid.replace('_elements_list', '');

        Array.from(elements).forEach(function(el) {
            el.dataset.step = stepid;
            el.dataset.idx = count;

            var reorder = {
                id: el.dataset.id,
                booste_step: stepid,
                idx: count
            };

            if(target.id != reorder.id ||
                (target.id == reorder.id && target.step == reorder.booste_step && target.idx == reorder.idx)) {
                reorderlist.push(reorder);
                count++;
           }
        });
    });

    const request = {
        methodname: 'local_booste_reorder_elements',
        args: {
            list: JSON.stringify(reorderlist)
        }
    };
    Ajax.call([request])[0].done(function(){
    }).catch(Notification.exception);

};

export const thirdFormLoader = () => {
      $(document).ajaxStop(function() {
        $("#booste_loader").hide();
      });
};

// export const refreshField = (trigger, changers) => {
//     console.log(changers);
//     $("#id_"+trigger).change(function() {
//         console.log($(this));
//         changers.forEach(e => {
//             $("#id_"+e)
//             .find('option')
//             .remove()
//             .end()
//             .append('<option value="whatever">text</option>')
//             .val('whatever')
//         });
//     });
// }