<?php

// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.
namespace local_booste;

use context_course;
use moodle_exception;

global $CFG;
require_once($CFG->libdir . '/externallib.php');

/**
 * Helper Main class (Lib of php funcs for booste)
 *
 * @package    local_booste
 * @copyright  XXX
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

class helper {

    /**
     * Header for main forms.
     *
     * @param int $courseid
     * @param int $boosteid
     * @return array[]
     */
    static function header($courseid, $boosteid) {

        global $PAGE;
        global $DB;

        $context = \context_course::instance($courseid);
        if (!has_capability('moodle/course:manageactivities', $context)) {
            throw new moodle_exception(get_string('no_access_to_booste', 'local_booste'));
        }
        if (!$courseid && !$boosteid) {
            throw new moodle_exception('noid');
        }
        if(!$courseid) {
            $booste = \local_booste\booste::get_record(['id' => $boosteid]);
            $courseid = $booste->get('course');
        }
        $course = $DB->get_record('course', array('id' => $courseid), '*', MUST_EXIST);
        require_course_login($course);

        $coursecontext = context_course::instance($course->id);

        $PAGE->set_title(format_string($course->fullname));
        $PAGE->set_heading(format_string($course->fullname));
        $PAGE->set_context($coursecontext);

        return ["course" => $courseid, "booste" => $boosteid];

    }

    /**
     * Access to protected functions.
     *
     * @param \stdClass $form
     * @param int $boosteid
     * @return \stdClass
     */
     static function access_protected(&$form, $boosteid) 

    {
        $reflection = new \ReflectionClass($form);
        $property = $reflection->getProperty("_form");
        $property->setAccessible(true);
        return $property->getValue($form);
    }
    

    /**
     * Check if user can edit forms.
     *
     * @param int $boosteid
     * @return bool
     */
    static function can_edit_form($boosteid) {
        global $USER;
        global $DB;

        if($boosteid) {
            $booste = $DB->get_record('local_booste', array('id' => $boosteid), '*', MUST_EXIST);
            if($USER->id == $booste->created_by || is_siteadmin()) {
                return true;
            }
            else {
                return false;
            }
        }
        else {
            return true;
        }
    }

    /**
     * Get tags of a certain field.
     *
     * @param string $field
     * @return array[]
     */
     static function get_records_from_field($field) {
        global $DB;
        $field_order = '';
        switch($field) {
            case "course_time":
                $field_order = "ORDER BY FIELD(name, 'course_under_60_mn', 'course_between_1h_and_1h30', 'course_between_1h_and_2h', 'course_over_2h')";
                break;
            case "taxonomy":
                $field_order = "ORDER BY FIELD(name, 'knowledge', 'comprehension', 'application', 'analysis', 'evaluation', 'synthesis')";
                break;
        }
        $records = $DB->get_records_sql_menu('SELECT id, name FROM {local_booste_list_tags} WHERE field = (SELECT id FROM {local_booste_list_fields} WHERE name = (?)) '.$field_order, array($field));
        foreach($records as $key => $value) {
            $records[$key] = get_string($value, 'local_booste');
            $records[$key] = str_replace(array('[',']'), '',$records[$key]);
        }
        return $records;
    }

    /**
     * Get tags from a booste of a certain field.
     *
     * @param int $boosteid
     * @param string $field
     * @return array[]
     */
     static function get_records_for_edit($boosteid, $field) {
        global $DB;
        $records = $DB->get_fieldset_sql(
            'SELECT id FROM {local_booste_list_tags} 
            WHERE field = (SELECT id FROM {local_booste_list_fields} WHERE name = (?))
            AND id IN (SELECT tag FROM {local_booste_tags} WHERE booste = (?))', array($field, $boosteid));
        return $records;
    }

    /**
     * Associate tags with a booste with fields parametered.
     *
     * @param int $field
     * @param object $field
     * @param array $field
     */
     static function create_booste_tags($boosteid, $fromform, $fields) {
        global $DB;

        foreach($fields as $field) {
            if(isset($fromform->$field)) {
                if(is_array($fromform->$field)) {
                    foreach($fromform->$field as $second_field) {
                        if($second_field && $second_field != "0") {
                            $DB->insert_record("local_booste_tags", array('booste' => $boosteid, 'tag' => $second_field));
                        }
                    }
                }
                else {
                    if($fromform->$field && $fromform->$field != "0") {
                        $DB->insert_record('local_booste_tags', array('booste' => $boosteid, 'tag' => $fromform->$field));
                    }
                }
            }
        }
    }

    /**
     * Delete tags of a booste with fields parametered.
     *
     * @param int $boosteid
     * @param array $fields
     */
     static function delete_booste_tags($boosteid, $fields) {
        global $DB;
        $fields_request = '(';
        foreach($fields as $field) {
            $fields_request .= "?,";
        }
        $fields_request = substr_replace($fields_request, ')', -1);
        array_unshift($fields, $boosteid);
        $DB->execute(
            "DELETE from {local_booste_tags} WHERE booste = (?) AND tag IN 
            (SELECT id FROM {local_booste_list_tags} WHERE field IN 
            (SELECT id FROM {local_booste_list_fields} WHERE name IN $fields_request)) AND is_taxonomy IS NULL", $fields);
    }


    /**
     * Get tags.
     *
     * @param int $boosteid
     * @return array[]
     */
     static function get_tags() {
        global $DB;
        $tags = $DB->get_records_menu('local_booste_list_tags', null, '', "id, name");
        foreach($tags as $key => $tag) {
            $tags[$key] = get_string($tag, 'local_booste');
            $tags[$key] = str_replace(array('[',']'), '',$tags[$key]);
        }
        return $tags;
    }

    /**
     * Get tags with fields.
     *
     * @param int $boosteid
     * @return array[]
     */
    static function get_tags_with_fields() {
        global $DB;
        $tag_records = $DB->get_records_sql("SELECT lblt.id AS id, lblf.name AS fieldname, lblt.name AS tagname 
                                    FROM {local_booste_list_tags} lblt
                                    LEFT JOIN {local_booste_list_fields} lblf ON lblt.field = lblf.id");
        $tags = [];
        foreach($tag_records as $record) {
            $record->tagname = get_string($record->tagname, 'local_booste');
            $record->tagname = str_replace(array('[',']'), '',$record->tagname);

            if($record->fieldname) {
                $record->fieldname = get_string($record->fieldname, 'local_booste');
                $record->fieldname = str_replace(array('[',']'), '',$record->fieldname);
                $tags[$record->id] = $record->fieldname." : ".$record->tagname;
            }
            else {
                $tags[$record->id] = $record->tagname;
            }
        }
        asort($tags);
        return $tags;
    }

    /**
     * Get booste tags.
     *
     * @param int $boosteid
     * @return array[]
     */
    static function get_booste_tags($boosteid) {
        global $DB;
        $records = $DB->get_records_sql_menu(
            'SELECT lblt.id, CONCAT(lblf.name,"&:",lblt.name) AS tag_name FROM {local_booste_list_tags} lblt
            LEFT JOIN {local_booste_list_fields} lblf ON lblf.id = lblt.field
            WHERE lblf.name <> "discipline"
            AND lblt.id IN (SELECT tag FROM {local_booste_tags} WHERE booste = (?))', array($boosteid));

        foreach($records as $key => $value) {
            $tstring = explode("&:", $value);
            $records[$key] = get_string($tstring[0], 'local_booste')." : ".get_string($tstring[1], 'local_booste');
            $records[$key] = str_replace(array('[',']'), '',$records[$key]);
        }
        return $records;
    }

    /**
     * Calculate tags points from the scenario to get its total points.
     *
     * @param int $boosteid
     * @param string $taxonomy
     * @return int
     */
     static function get_booste_points($boosteid, $taxonomy) {
        global $DB;

        if($taxonomy == "all") {
            $taxonomies = "";
        }
        else {
            $taxonomies = "AND lblt.name NOT IN ('evaluation', 'synthesis', 'analysis', 'application', 'comprehension', 'knowledge') ";
            $taxonomies = str_replace($taxonomy, '', $taxonomies);
        }
        $sql = 'SELECT SUM(points)
                FROM mdl_local_booste_list_tags lblt 
                LEFT JOIN mdl_local_booste_list_fields lblf  ON lblt.field = lblf.id
                LEFT JOIN mdl_local_booste_tags lbt  ON lbt.tag = lblt.id
                WHERE
                lbt.booste = (?)
                AND lblf.name NOT IN ("discipline", "course_time")
                AND lblt.ignore_suggestions = 0'; 

        $points = $DB->get_field_sql($sql, array($boosteid, $boosteid)) + 10;

        return $points;
    }

    /**
     * Create a module in course.
     *
     * @param object $stepmodule
     * @param object $oldcm
     * @param object $course
     * @param int $section
     * @return object
     */
     static function create_module($stepmodule, $oldcm, $course, $section) {
        global $CFG;
        $moduledata = json_decode(htmlspecialchars_decode($stepmodule->data));

        foreach($moduledata as $key => $value) {
            if(is_object($value)) {
                $moduledata->$key = (array) $moduledata->$key;
                if(is_array($value) && array_key_exists("text", $value)) {
                    $moduledata->$key['text'] = html_entity_decode($moduledata->$key['text']);
                }
            }
            if(is_array($value) && count($value) > 0) {
                foreach($value as $key2 => $value2) {
                    if(is_object($value2)) {
                        $value[$key2] = (array) $value2;
                    }
                }
                $moduledata->$key = $value;
            }
        }
        
        list($module, $context, $cw, $cm, $data) = prepare_new_moduleinfo_data($course, $oldcm->name, $moduledata->section);
        if($data->modulename == "game") {
            $_POST['type'] = $moduledata->type;
        }
        $modmoodleform = "$CFG->dirroot/mod/$module->name/mod_form.php";
        if (file_exists($modmoodleform)) {
            require_once($modmoodleform);
        } else {
            print_error('noformdesc');
        }
        $mformclassname = 'mod_'.$module->name.'_mod_form';
        $mform = new $mformclassname($data, $cw->section, $cm, $course);

        $moduledata->section = $section;
        $data->section = $section;

        $mform->set_data($data);

        return add_moduleinfo($moduledata, $course, $mform);
    }

    // function fields_helper($fieldname) {
    //     global $DB;
    //     $field = $DB->get_record("local_booste_list_fields", array("name" => $fieldname));
    //     if($field->helper) {
    //         return $field->helper;
    //     }
    //     else {
    //         return get_string($field->helper, "local_booste");
    //     }

    // }
}
