// This file is part of Moodle - https://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <https://www.gnu.org/licenses/>.import Ajax from 'core/ajax';

/*
 * All pop ups for booste
 * @module     local_booste/popup_form
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
import Ajax from 'core/ajax';
import JQuery from 'jquery';
import ModalForm from 'core_form/modalform';
import {get_string as getString} from 'core/str';
import {refreshBoosteDatas} from "./refresh_data";
import Notification from 'core/notification';

/**
 * Init the creation or the edit of an activity, objective, guideline, resource or a step
 *
 * @method init
 * @param {String} form type of element to generate the form
 */

export const init = (form, isNew) => {

    if(isNew) {
        document.querySelector('[data-action=newboostepopup][data-formname='+form+']')
            .addEventListener('click', (e) => setPopup(e));
    }
    else {
        let popups = document.querySelectorAll('[data-action=editboostepopup][data-formname='+form+']');
        popups.forEach(popup => {
            popup.addEventListener('click', (e) => setPopup(e));
        });
    }
};

/**
 * Setting the popup depending of the type of form specified in the element
 *
 * @method init
 * @param {String} e event
 */
const setPopup = (e) => {
    e.preventDefault();
    var element;
    var url = window.location.href;
    var urlParams = new URLSearchParams(window.location.search);
    var boosteid = urlParams.get('booste') ? urlParams.get('booste') : 0;
    var courseid = urlParams.get('course') ? urlParams.get('course') : 0;

    if(e.target.tagName == "I") {
        element = e.currentTarget;
    }
    else {
        element = e.target;
    }
    const id = element.dataset.id ? element.dataset.id : 0;
    const editid = element.dataset.editid ? element.dataset.editid : 0;
    const formName = element.id.substring(3);
    const specForms = ["objective", "booste_step", "activities_tags", "ext_activity", "tag", "evaluation", "challenge", "field"];

    if(specForms.includes(formName)) {
        if(formName == "challenge") {
            var challengedata = {
                challenge: $("#id_challenge option:selected").toArray().map( o => o.value ),
                learning_mode: $("[name=learning_mode]:checked").val()
            };
            challengedata = JSON.stringify(challengedata);
        }
        else {
            var challengedata = [];
        }
        const modalForm = new ModalForm({
            // Name of the class where form is defined (must extend \core_form\dynamic_form):
            formClass: "local_booste\\form\\"+formName+"_form",
            // Add as many arguments as you need, they will be passed to the form:
            args: {id: id, editid: editid, courseid: courseid, challengedata: challengedata},
            // Pass any configuration settings to the modal dialogue, for example, the title:
            modalConfig: {title: getString(formName, 'local_booste')},
            // DOM element that should get the focus after the modal dialogue is closed:
            returnFocus: element,
        });
        // Listen to events if you want to execute something on form submit.
        // Event detail will contain everything the process() function returned:
        modalForm.addEventListener(modalForm.events.FORM_SUBMITTED, () => {
            switch(formName) {
                case "objective":
                case "booste_step":
                case "evaluation":
                    refreshBoosteDatas(id, formName);
                    break;
                case "tag":
                case "activities_tags":
                case "ext_activity":
                case "field":
                    document.location.reload();
                    break;
                case "challenge":
                    window.location = window.location.href;
                    break;
                default:
                    break;
            }
        });
        // Show the form.
        modalForm.show();
    }
    else {
        const request = {
            methodname: 'local_booste_get_module',
            args: {
                id: element.dataset.editid
            }
        };
        Ajax.call([request])[0].done(function(context) {
            let data = JSON.parse(context.data);
            var type = '';
            url = url.split("booste_third_form")[0];
            if(data.type) {
                type = "&type="+data.type;
            }
            url += "modedit.php?course="+courseid+"&booste="+boosteid+type+"&update="+element.dataset.editid+"&booste_step="+id;
            window.location.href = url;

        }).catch(Notification.exception);
    }
};

/**
 * Deploy all activities/resources in the actual course
 *
 * @method init
 * @param {Number} booste id of actual booste
 * @param {Number} courseid
 */

export const deployBooste = (booste, courseid) => {

    document.getElementById("id_deploy").classList.remove("btn-secondary");
    document.getElementById("id_deploy").classList.add("btn-primary");
    document.getElementById("id_deploy").addEventListener('click', () => {
        document.querySelector('#page-header').scrollIntoView();
        var isDeployable = 1;
        // if(booste.is_deployed == 1) {
        //     isDeployable = 0;
        // }
        if(isDeployable == 1) {
            const request1 = {
                methodname: 'local_booste_get_booste_steps',
                args: {
                    id: booste.id
                }
            };
            Ajax.call([request1])[0].done(function(context1) {
                var elements = [];
                context1.forEach(function(item) {
                    const request2 = {
                        methodname: 'local_booste_get_elements',
                        args: {
                            id: item.id
                        }
                    };
                    elements.push(Ajax.call([request2])[0].done(function(context2) {
                        if(context2.length == 0) {
                            isDeployable = 0;
                        }
                    }).catch(Notification.exception));
                });

                JQuery.when(...elements).then(function() {
                    if(isDeployable == 1) {
                        const modalForm = new ModalForm({
                            // Name of the class where form is defined (must extend \core_form\dynamic_form):
                            formClass: "local_booste\\form\\booste_fourth_form",
                            // Add as many arguments as you need, they will be passed to the form:
                            args: {booste: booste.id, course: courseid},
                            // Pass any configuration settings to the modal dialogue, for example, the title:
                            modalConfig: {title: getString('deploy', 'local_booste'),
                                            buttons: [getString('booste_deploy', 'local_booste')]},
                            // DOM element that should get the focus after the modal dialogue is closed:
                            returnFocus: null,
                        });
                        modalForm.addEventListener(modalForm.events.FORM_SUBMITTED, (e) => {
                            var url = window.location.href;
                            url = url.split("booste_third_form")[0];
                            url += "deploy.php?courseid="+e.detail.course+"&boosteid="+e.detail.booste+"&deploy="+e.detail.deploy;
                            window.location.href = url;
                        });
                        modalForm.show();
                    } else {
                        getString('elements_missing', 'local_booste')
                        .then(str => {
                            Notification.addNotification({
                                type: "error",
                                message: str
                            });
                            return;
                        })
                        .catch();
                    }
                });
            }).catch(Notification.exception);
        }
        else {
            getString('already_deployed', 'local_booste')
            .then(str => {
                Notification.addNotification({
                    type: "error",
                    message: str
                });
                return;
            })
            .catch();
        }
    });
};

/**
 * Duplicate a scenario
 *
 * @method duplicateBooste
 * @param {Number} id
 */
export async function duplicateBooste(id)  {
    document.querySelector('[data-action="duplicatebooste"][data-id="'+id+'"]').addEventListener('click', (e) => {
        let course = e.target.dataset.course;
        const modalForm = new ModalForm({
            // Name of the class where form is defined (must extend \core_form\dynamic_form):
            formClass: "local_booste\\form\\booste_duplicate_form",
            // Add as many arguments as you need, they will be passed to the form:
            args: {id: id, course: course},
            // Pass any configuration settings to the modal dialogue, for example, the title:
            modalConfig: {title: getString('booste_duplicate', 'local_booste'), buttons: [M.util.get_string('confirm', 'moodle')]},
            // DOM element that should get the focus after the modal dialogue is closed:
            returnFocus: null,
        });
        modalForm.addEventListener(modalForm.events.FORM_SUBMITTED, (e) => {
            getString('duplicated', 'local_booste')
            .then(str => {
                document.querySelector('#page-header').scrollIntoView();
                str = str.replace("<a", "<a href='"+e.detail+"' ");
                Notification.addNotification({
                    type: "success",
                    message: str
                });
                return;
            })
            .catch();
        });
        modalForm.show();
    });
}